"use client";
import { useEffect, useState, FormEvent } from "react";
import Link from "next/link";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  ArrowLeft,
  Users,
  Loader2,
  Wrench,
  CheckCircle2,
  AlertCircle,
  RefreshCw,
  Save,
  Search,
  Filter,
  Settings,
  MoreVertical,
} from "lucide-react";
import EditTechnicianDialog from "@/components/EditTechnicianDialog";
import CreateTechnicianDialog from "@/components/CreateTechnicianDialog";
import DeleteTechnicianDialog from "@/components/DeleteTechnicianDialog";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

type User = {
  id: string;
  email: string;
  name?: string | null;
  role: string;
  hourlyRate?: number | null;
  isExternal?: boolean;
  companyName?: string | null;
  customRole?: {
    id: string;
    name: string;
    description?: string | null;
  } | null;
  createdAt: string;
};

type Role = {
  id: string;
  name: string;
  description?: string | null;
  isActive: boolean;
};

export default function AdminTechnicians() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const role = String((session?.user as any)?.role || "").toUpperCase();
  const isAdmin = role === "ADMIN";

  const [rows, setRows] = useState<User[]>([]);
  const [roles, setRoles] = useState<Role[]>([]);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);

  // Filter states - default to TECHNICIAN role
  const [filters, setFilters] = useState({
    role: "TECHNICIAN",
    search: "",
  });
  const [showFilters, setShowFilters] = useState(false);

  // No longer need create form state (now in dialog)

  // feedback
  const [msg, setMsg] = useState<string | null>(null);

  // role assignment
  const [assigningRole, setAssigningRole] = useState<string | null>(null);

  // edit dialog state
  const [showEditDialog, setShowEditDialog] = useState(false);
  const [editingTechnician, setEditingTechnician] = useState<any>(null);

  // create dialog state
  const [showCreateDialog, setShowCreateDialog] = useState(false);

  // delete dialog state
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [deletingTechnician, setDeletingTechnician] = useState<any>(null);

  // Track dropdown state to force remount
  const [dropdownKey, setDropdownKey] = useState(0);

  const fetchList = () => {
    setLoading(true);
    setErr(null);

    // Build query string from filters
    const queryParams = new URLSearchParams();
    if (filters.role) queryParams.append("role", filters.role);
    if (filters.search) queryParams.append("search", filters.search);

    const queryString = queryParams.toString();
    const url = queryString
      ? `/api/admin/users?${queryString}`
      : "/api/admin/users";

    fetch(url)
      .then((r) => r.json())
      .then((j) => {
        if (!j.ok) throw new Error(j.error || "failed");
        setRows(j.users || j.technicians || []);
      })
      .catch((e) => setErr(String(e)))
      .finally(() => setLoading(false));
  };

  const fetchRoles = () => {
    fetch("/api/admin/roles?isActive=true")
      .then((r) => r.json())
      .then((j) => {
        if (j.ok) setRoles(j.roles || []);
      })
      .catch(() => {}); // Silently fail for roles
  };

  useEffect(() => {
    fetchList();
    fetchRoles();
  }, [filters]);

  const onAssignRole = async (userId: string, roleId: string | null) => {
    setAssigningRole(userId);
    setErr(null);
    setMsg(null);
    try {
      const resp = await fetch("/api/admin/roles/assign", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ userId, roleId }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Assignment failed");
      setMsg(json.message || "Role assignment updated.");
      fetchList();
    } catch (e: any) {
      setErr(String(e?.message || e));
    } finally {
      setAssigningRole(null);
    }
  };

  // edit dialog functions
  const openEditDialog = (technician: any) => {
    setEditingTechnician(technician);
    setShowEditDialog(true);
    // Force dropdown remount by changing key
    setDropdownKey((prev) => prev + 1);
  };

  const closeEditDialog = () => {
    setShowEditDialog(false);
    setEditingTechnician(null);
  };

  const handleSaveEdit = async (data: {
    name: string;
    email: string;
    hourlyRate: number | null;
    customRoleId: string | null;
    isExternal: boolean;
    companyName: string | null;
  }) => {
    if (!editingTechnician) return;

    setErr(null);
    setMsg(null);

    const resp = await fetch(
      `/api/admin/users/${editingTechnician.id}/update`,
      {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: data.name || null,
          email: data.email,
          hourlyRate: data.hourlyRate,
          customRoleId: data.customRoleId,
          isExternal: data.isExternal,
          companyName: data.companyName,
        }),
      }
    );
    const json = await resp.json();
    if (!json.ok) throw new Error(json.error || "Update failed");
    setMsg("Technician updated successfully.");
    fetchList();
  };

  const handleCreate = async (data: {
    email: string;
    name: string;
    hourlyRate: number | null;
    customRoleId: string | null;
    isExternal: boolean;
    companyName: string | null;
  }) => {
    setErr(null);
    setMsg(null);

    const resp = await fetch("/api/admin/users/create", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        email: data.email,
        name: data.name,
        role: "TECHNICIAN",
        hourlyRate: data.hourlyRate,
        customRoleId: data.customRoleId,
        isExternal: data.isExternal,
        companyName: data.companyName,
      }),
    });
    const json = await resp.json();
    if (!json.ok) throw new Error(json.error || "Create failed");
    setMsg("Technician created successfully. A welcome email with password setup link has been sent.");
    fetchList();
  };

  // delete dialog functions
  const openDeleteDialog = (technician: any) => {
    setDeletingTechnician(technician);
    setShowDeleteDialog(true);
    // Force dropdown remount by changing key
    setDropdownKey((prev) => prev + 1);
  };

  const closeDeleteDialog = () => {
    setShowDeleteDialog(false);
    setDeletingTechnician(null);
  };

  const handleDeleteTechnician = async (unassignFromTasks: boolean = false, unassignFromTickets: boolean = false) => {
    if (!deletingTechnician) return;

    setErr(null);
    setMsg(null);

    try {
      const params = new URLSearchParams();
      if (unassignFromTasks) params.append("unassignFromTasks", "true");
      if (unassignFromTickets) params.append("unassignFromTickets", "true");
      const queryString = params.toString();
      const url = `/api/admin/users/${deletingTechnician.id}${queryString ? `?${queryString}` : ""}`;
      const resp = await fetch(url, {
        method: "DELETE",
      });
      const json = await resp.json();
      if (!json.ok) {
        // Create error object with additional metadata
        const error: any = new Error(json.error || "Delete failed");
        error.canUnassign = json.canUnassign;
        error.assignedTasksCount = json.assignedTasksCount;
        error.assignedTasks = json.assignedTasks;
        error.canUnassignTickets = json.canUnassignTickets;
        error.assignedTicketsCount = json.assignedTicketsCount;
        error.assignedTickets = json.assignedTickets;
        throw error;
      }
      setMsg(json.message || "Technician deleted successfully.");
      fetchList();
    } catch (e: any) {
      setErr(String(e?.message || e));
      throw e; // Re-throw to handle in dialog
    }
  };

  if (status === "loading") {
    return <main className='max-w-5xl mx-auto px-4 py-6'>Loading…</main>;
  }
  if (!isAdmin) {
    return (
      <main className='max-w-5xl mx-auto px-4 py-6'>
        <h1 className='text-2xl font-bold'>Manage Technicians</h1>
        <div className='alert alert-error mt-3'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>Forbidden — admin only</p>
            <p className='text-sm opacity-90'>
              You don’t have permission to view this page.
            </p>
          </div>
        </div>
      </main>
    );
  }

  return (
    <main className='w-full p-6'>
      {/* Header */}
      <div className='mb-4 flex items-center justify-between'>
        <div>
          <Link
            href='/'
            className='inline-flex items-center gap-2 text-sm text-gray-600 hover:text-gray-900 mb-2'
          >
            <ArrowLeft className='w-4 h-4' />
            Back to Dashboard
          </Link>
          <h1 className='text-2xl font-bold text-gray-900'>
            Manage Technicians
          </h1>
          <p className='text-gray-600 mt-1 text-sm'>
            Create and manage technician accounts
          </p>
        </div>
        <button
          onClick={() => setShowCreateDialog(true)}
          className='px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 inline-flex items-center gap-2'
        >
          <Save className='w-4 h-4' />
          {t("admin.create")} {t("admin.technicians")}
        </button>
      </div>

      {/* Feedback */}
      {msg && (
        <div className='mb-6 p-4 bg-green-50 border border-green-200 rounded-md'>
          <p className='text-green-700'>{msg}</p>
        </div>
      )}
      {err && (
        <div className='mb-6 p-4 bg-red-50 border border-red-200 rounded-md'>
          <p className='text-red-700'>{err}</p>
        </div>
      )}

      {/* Technicians List */}
      <div className='bg-white border border-gray-200 rounded-lg p-4'>
        <div className='flex items-center justify-between mb-3'>
          <h2 className='text-lg font-medium text-gray-900'>
            {t("admin.technicians")} ({rows.length})
          </h2>
          <div className='flex items-center gap-3'>
            <button
              onClick={() => setShowFilters(!showFilters)}
              className={`px-3 py-2 text-sm border rounded-md hover:bg-gray-50 ${
                showFilters || Object.values(filters).some((v) => v)
                  ? "bg-blue-50 border-blue-300 text-blue-700"
                  : "border-gray-300"
              }`}
            >
              <Filter className='w-4 h-4 inline mr-1' />
              {t("tickets.filters")}
            </button>
          </div>
        </div>

        {/* Filter Panel */}
        {showFilters && (
          <div className='mb-3 p-3 bg-gray-50 rounded-lg border border-gray-200'>
            <div className='grid grid-cols-1 md:grid-cols-2 gap-4'>
              <div>
                <label className='block text-sm font-medium text-gray-700 mb-2'>
                  {t("common.search")}
                </label>
                <div className='relative'>
                  <Search className='absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400' />
                  <input
                    type='text'
                    placeholder={
                      t("admin.searchTechnicians") || "Search technicians..."
                    }
                    value={filters.search}
                    onChange={(e) =>
                      setFilters((prev) => ({
                        ...prev,
                        search: e.target.value,
                      }))
                    }
                    className='w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent'
                  />
                </div>
              </div>

              <div>
                <label className='block text-sm font-medium text-gray-700 mb-2'>
                  {t("sidebar.roles") || "Role"}
                </label>
                <select
                  value={filters.role}
                  onChange={(e) =>
                    setFilters((prev) => ({ ...prev, role: e.target.value }))
                  }
                  className='w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent'
                >
                  <option value=''>{t("admin.allRoles") || "All Roles"}</option>
                  <option value='ADMIN'>Admin</option>
                  <option value='TECHNICIAN'>{t("admin.technicians")}</option>
                  <option value='USER'>User</option>
                </select>
              </div>
            </div>
          </div>
        )}

        {loading ? (
          <div className='flex items-center justify-center py-8'>
            <Loader2 className='w-6 h-6 animate-spin text-gray-400 mr-3' />
            <span className='text-gray-600'>
              {t("admin.loadingTechnicians") || "Loading technicians..."}
            </span>
          </div>
        ) : rows.length === 0 ? (
          <div className='text-center py-8'>
            <Users className='w-12 h-12 text-gray-400 mx-auto mb-3' />
            <p className='text-gray-600'>
              {t("admin.noTechnicians") || "No technicians found"}
            </p>
          </div>
        ) : (
          <div className='overflow-x-auto'>
            <table className='w-full text-sm'>
              <thead className='bg-gray-50 border-b border-gray-200'>
                <tr>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    {t("common.name") || "Name"}
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    Email
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    Role
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    Hourly Rate
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    External
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    {t("admin.companyName") || "Company"}
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    {t("tasks.detail.created") || "Created"}
                  </th>
                  <th className='text-left font-medium text-gray-700 px-3 py-2'>
                    {t("common.actions") || "Actions"}
                  </th>
                </tr>
              </thead>
              <tbody className='divide-y divide-gray-200'>
                {rows.map((u) => (
                  <tr key={u.id} className='hover:bg-gray-50'>
                    <td className='px-3 py-2'>
                      <div className='font-medium text-gray-900'>
                        {u.name || "—"}
                      </div>
                      <div className='text-xs text-gray-500 mt-0.5'>
                        {u.role}
                      </div>
                    </td>
                    <td className='px-3 py-2 text-gray-600'>{u.email}</td>
                    <td className='px-3 py-2'>
                      {u.customRole?.name ? (
                        <span className='inline-flex px-2 py-0.5 text-xs font-semibold rounded-full bg-blue-100 text-blue-800'>
                          {u.customRole.name}
                        </span>
                      ) : (
                        <span className='text-gray-400 text-sm'>—</span>
                      )}
                    </td>
                    <td className='px-3 py-2 text-gray-600'>
                      {u.hourlyRate ? `€${u.hourlyRate}/hr` : "Not set"}
                    </td>
                    <td className='px-3 py-2'>
                      {u.isExternal ? (
                        <span className='inline-flex px-2 py-0.5 text-xs font-semibold rounded-full bg-orange-100 text-orange-800'>
                          External
                        </span>
                      ) : (
                        <span className='text-gray-400 text-sm'>—</span>
                      )}
                    </td>
                    <td className='px-3 py-2 text-gray-600'>
                      {u.companyName || "—"}
                    </td>
                    <td className='px-3 py-2 text-gray-500'>
                      {new Date(u.createdAt).toLocaleDateString()}
                    </td>
                    <td className='px-3 py-2'>
                      {u.role?.toUpperCase() === "TECHNICIAN" ? (
                        <DropdownMenu key={`${u.id}-${dropdownKey}`}>
                          <DropdownMenuTrigger asChild>
                            <button
                              className='p-1 text-gray-400 hover:text-gray-600 rounded-full hover:bg-gray-100'
                              title='More actions'
                            >
                              <MoreVertical className='w-4 h-4' />
                            </button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align='end' className='w-48'>
                            <DropdownMenuItem onClick={() => openEditDialog(u)}>
                              Edit Details
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() => openDeleteDialog(u)}
                              className='text-red-600 focus:text-red-600 focus:bg-red-50'
                            >
                              Delete Technician
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      ) : (
                        <span className='text-gray-400 text-sm'>—</span>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Edit Technician Dialog */}
      <EditTechnicianDialog
        isOpen={showEditDialog}
        technician={editingTechnician}
        roles={roles}
        onClose={closeEditDialog}
        onSave={handleSaveEdit}
      />

      {/* Create Technician Dialog */}
      <CreateTechnicianDialog
        isOpen={showCreateDialog}
        roles={roles}
        onCreate={handleCreate}
        onClose={() => setShowCreateDialog(false)}
      />

      {/* Delete Technician Dialog */}
      <DeleteTechnicianDialog
        isOpen={showDeleteDialog}
        technicianName={deletingTechnician?.name}
        technicianEmail={deletingTechnician?.email || ""}
        onClose={closeDeleteDialog}
        onConfirm={handleDeleteTechnician}
      />
    </main>
  );
}
